<?php

/**
 * This class should be used to stores properties and methods shared by the
 * admin and public side of WordPress.
 */
class Daln_Shared {

	/**
	 * Regex.
	 *
	 * @var string
	 */
	public $hex_rgb_regex     = '/^#(?:[0-9a-fA-F]{3}){1,2}$/';
	public $font_family_regex = '/^([A-Za-z0-9-\'", ]*)$/';
	public $url_regex         = '/^(http|https):\/\/[-A-Za-z0-9+&@#\/%?=~_|$!:,.;]+$/';
	public $regex_capability  = '/^\s*[A-Za-z0-9_]+\s*$/';

	protected static $instance = null;

	private $data = array();

	private function __construct() {

		// Set plugin textdomain.
		load_plugin_textdomain( 'daln', false, 'live-news/lang/' );

		$this->data['slug'] = 'daln';
		$this->data['ver']  = '2.17';
		$this->data['dir']  = substr( plugin_dir_path( __FILE__ ), 0, - 7 );
		$this->data['url']  = substr( plugin_dir_url( __FILE__ ), 0, - 7 );

		// Here are stored the plugin option with the related default values.
		$this->data['options'] = array(

			// Database Version ---------------------------------------------------------------------------------------.
			$this->get( 'slug' ) . '_database_version' => '0',

			// General ------------------------------------------------------------------------------------------------.
			$this->get( 'slug' ) . '_detect_url_mode'  => 'server_variable',
			$this->get( 'slug' ) . '_load_momentjs'    => '1',
			$this->get( 'slug' ) . '_tickers_menu_capability' => 'manage_options',
			$this->get( 'slug' ) . '_featured_menu_capability' => 'manage_options',
			$this->get( 'slug' ) . '_sliding_menu_capability' => 'manage_options',
			$this->get( 'slug' ) . '_import_menu_capability' => 'manage_options',
			$this->get( 'slug' ) . '_export_menu_capability' => 'manage_options',
			$this->get( 'slug' ) . '_maintenance_menu_capability' => 'manage_options',

		);
	}

	public static function get_instance() {

		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Retrieve data.
	 *
	 * @param int $index The index.
	 *
	 * @return mixed
	 */
	public function get( $index ) {
		return $this->data[ $index ];
	}

	/**
	 *
	 *  Convert a numeric target to a textual target
	 *
	 * @param int $target_id The target id.
	 *
	 * @return string|void
	 */
	public function get_textual_target( $target_id ) {

		switch ( $target_id ) {

			case 1:
				return 'Website';

			case 2:
				return 'URL';

		}
	}

	/**
	 * Convert a numeric source to a textual source
	 *
	 * @param int $source_id The source id.
	 *
	 * @return string
	 */
	public function get_textual_source( $source_id ) {

		switch ( $source_id ) {

			case 1:
				return __( 'Manually Added', 'daln' );

			case 2:
				return __( 'Posts', 'daln' );

			case 3:
				return __( 'RSS', 'daln' );

			case 4:
				return __( 'Twitter', 'daln' );

		}
	}

	/**
	 * Retrieve the ticker name from the ticker id
	 *
	 * @param int $ticker_id The ticker id.
	 *
	 * @return string
	 */
	public function get_textual_ticker( $ticker_id ) {

		global $wpdb;
		$table_name = $wpdb->prefix . $this->get( 'slug' ) . '_tickers';
		$safe_sql   = $wpdb->prepare( "SELECT name FROM $table_name WHERE id = %d ", $ticker_id );
		$ticker_obj = $wpdb->get_row( $safe_sql );

		if ( null !== $ticker_obj ) {
			return $ticker_obj->name;
		} else {
			return 'Invalid Ticker ID';
		}
	}

	/**
	 * Generate a short version of a string without truncating words
	 *
	 * @param String $str The string.
	 * @param Int    $length The maximum length of the string.
	 *
	 * @return string The short version of the string
	 */
	public function strlen_no_truncate( $str, $length ) {

		if ( mb_strlen( $str ) > $length ) {
			$str = wordwrap( $str, $length );
			$str = mb_substr( $str, 0, mb_strpos( $str, "\n" ) );
			$str = $str . ' ...';
		}

		return $str;
	}

	/**
	 * Returns true if the ticker is used in sliding news or in featured news
	 *
	 * @param Int $ticker_id The ticker id.
	 *
	 * @return bool True if the ticker is used or False if the ticker is not used
	 */
	public function ticker_is_used( $ticker_id ) {

		global $wpdb;

		// Verify if the ticker is used in the featured news.
		$table_name     = $wpdb->prefix . $this->get( 'slug' ) . '_featured_news';
		$safe_sql       = $wpdb->prepare( "SELECT COUNT(*) FROM $table_name WHERE ticker_id = %d", $ticker_id );
		$number_of_uses = $wpdb->get_var( $safe_sql );
		if ( $number_of_uses > 0 ) {
			return true;
		}

		// Verify if the ticker is used in the sliding news.
		$table_name     = $wpdb->prefix . $this->get( 'slug' ) . '_sliding_news';
		$safe_sql       = $wpdb->prepare( "SELECT COUNT(*) FROM $table_name WHERE ticker_id = %d", $ticker_id );
		$number_of_uses = $wpdb->get_var( $safe_sql );
		if ( $number_of_uses > 0 ) {
			return true;
		}

		return false;
	}

	/**
	 * Given a ticker id returns true if the ticker exists or false if the ticker doesn't exist
	 *
	 * @param Int $ticker_id The ticker id.
	 *
	 * @return bool True if the ticker exists or False if the ticker doesn't exists
	 */
	public function ticker_exists( $ticker_id ) {

		global $wpdb;

		$table_name = $wpdb->prefix . $this->get( 'slug' ) . '_tickers';
		$safe_sql   = $wpdb->prepare( "SELECT * FROM $table_name WHERE id = %d", $ticker_id );
		$ticker_obj = $wpdb->get_row( $safe_sql );
		if ( null !== $ticker_obj ) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Given a hexadecimal rgb color an array with the 3 components converted in decimal is returned
	 *
	 * @param string $hex The hexadecimal rgb color.
	 *
	 * @return array An array with the 3 component of the color converted in decimal
	 */
	public function rgb_hex_to_dec( $hex ) {

		if ( 3 === mb_strlen( $hex ) ) {
			$r = hexdec( substr( $hex, 0, 1 ) . substr( $hex, 0, 1 ) );
			$g = hexdec( substr( $hex, 1, 1 ) . substr( $hex, 1, 1 ) );
			$b = hexdec( substr( $hex, 2, 1 ) . substr( $hex, 2, 1 ) );
		} else {
			$r = hexdec( substr( $hex, 0, 2 ) );
			$g = hexdec( substr( $hex, 2, 2 ) );
			$b = hexdec( substr( $hex, 4, 2 ) );
		}

		return array(
			'r' => $r,
			'g' => $g,
			'b' => $b,
		);
	}

	/**
	 * Get the number of tickers
	 *
	 * @return int The number of tickers.
	 */
	public function get_number_of_tickers() {

		global $wpdb;
		$table_name        = $wpdb->prefix . $this->get( 'slug' ) . '_tickers';
		$number_of_tickers = $wpdb->get_var( "SELECT COUNT(*) FROM $table_name" );

		return intval($number_of_tickers, 10);
	}

	/**
	 * Get the number of featured news.
	 *
	 * @return string|null
	 */
	public function get_number_of_featured_news() {

		global $wpdb;
		$table_name              = $wpdb->prefix . $this->get( 'slug' ) . '_featured_news';
		$number_of_featured_news = $wpdb->get_var( "SELECT COUNT(*) FROM $table_name" );

		return $number_of_featured_news;
	}

	/**
	 * Get the number of sliding news.
	 *
	 * @return string|null The number of sliding news
	 */
	public function get_number_of_sliding_news() {

		global $wpdb;
		$table_name             = $wpdb->prefix . $this->get( 'slug' ) . '_sliding_news';
		$number_of_sliding_news = $wpdb->get_var( "SELECT COUNT(*) FROM $table_name" );

		return $number_of_sliding_news;
	}

	/**
	 * Based on the selected option this method is used to remove the following elements included in a tweet:
	 *
	 *  - links
	 *  - hashtags
	 *  - usernames
	 *
	 * @param $tweet string The tweet
	 * @param $strip_links bool Whether to strip links
	 * @param $strip_hashtags bool Whether to strip hashtags
	 * @param $strip_usernames bool Whether to strip usernames
	 *
	 * @return array|mixed|string|string[]|null The tweet with the elements removed
	 */
	public function twitter_remove_elements( $tweet, $strip_links, $strip_hashtags, $strip_usernames ) {

		// Strip links (generic url regex which allows almost any url).
		if ( $strip_links ) {
			$tweet = preg_replace( '/(https?|ftp|file):\/\/[^\s]+/', '', $tweet );
		}

		// Strip hashtags.
		if ( $strip_hashtags ) {
			$tweet = preg_replace( '/#[a-zA-Z_]{1}[a-zA-Z0-9_]*/', '', $tweet );
		}

		// Strip usernames.
		if ( $strip_usernames ) {
			$tweet = preg_replace( '/@[a-zA-Z0-9_]{1,15}/', '', $tweet );
		}

		return $tweet;
	}

	/**
	 * Get the current URL with the method specified with the "Detect URL Mode" option.
	 *
	 * @return string
	 */
	public function get_current_url() {

		if ( get_option( $this->get( 'slug' ) . '_detect_url_mode' ) === 'server_variable' ) {

			// Detect the URL using the "Server Variable" method.
			return is_ssl() ? 'https' : 'http' . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];

		} else {

			// Detect the URL using the "WP Request" method.
			global $wp;

			return trailingslashit( home_url( add_query_arg( array(), $wp->request ) ) );

		}
	}

	/**
	 * Returns the object of the first ticker with the target equal to 2 (url) that can be displayed with the current
	 * url.
	 *
	 * Note that to determine if a ticker can be displayed with the current url the following fields are considered:
	 *
	 * - Target
	 * - Target URL
	 * - Target URL Mode
	 *
	 * @param String $current_url The url that should be searched in the target url field of the tickers.
	 *
	 * @return mixed The object with the data of the news ticker of false.
	 */
	public function get_ticker_with_target_url( $current_url ) {

		$found      = false;
		$ticker_id  = null;
		$ticker_obj = false;

		global $wpdb;
		$table_name = $wpdb->prefix . $this->get( 'slug' ) . '_tickers';
		$safe_sql   = "SELECT * FROM $table_name WHERE target = 2 ORDER BY id ASC";
		$ticker_a   = $wpdb->get_results( $safe_sql, ARRAY_A );

		foreach ( $ticker_a as $key => $ticker ) {

			$url_a = preg_split( '/\r\n|[\r\n]/', $ticker['url'] );

			if ( intval( $ticker['url_mode'], 10 ) === 0 ) {

				// Include.

				// Get the ticker_id of the first news ticker that includes the current url.
				if ( null !== $ticker_id ) {
					break;
				}

				foreach ( $url_a as $key2 => $url ) {
					if ( $url === $current_url ) {
						$found = true;
					}
				}

				if ( $found ) {
					$ticker_id = $ticker['id'];
					break;
				}
			} else {

				// Exclude.

				// Get the ticker_id of the first news ticker that doesn't include the current url.
				foreach ( $url_a as $key2 => $url ) {
					if ( $url === $current_url ) {
						$found = true;
					}
				}

				if ( ! $found ) {
					$ticker_id = $ticker['id'];
					break;
				}
			}
		}

		// Get the object of the news ticker that includes the current url.
		if ( null !== $ticker_id ) {
			global $wpdb;
			$table_name = $wpdb->prefix . $this->get( 'slug' ) . '_tickers';
			$safe_sql   = $wpdb->prepare( "SELECT * FROM $table_name WHERE id = %d", $ticker_id );
			$ticker_obj = $wpdb->get_row( $safe_sql );
		}

		return $ticker_obj;
	}

	/**
	 * Generates the XML version of the data of the table.
	 *
	 * @param string $db_table_name The name of the db table without the prefix.
	 * @param string $db_table_primary_key The name of the primary key of the table.
	 *
	 * @return String The XML version of the data of the db table.
	 */
	public function convert_db_table_to_xml( $db_table_name, $db_table_primary_key ) {

		$out = '';

		// Get the data from the db table.
		global $wpdb;
		$table_name = $wpdb->prefix . $this->get( 'slug' ) . "_$db_table_name";
		$data_a     = $wpdb->get_results( "SELECT * FROM $table_name ORDER BY $db_table_primary_key ASC", ARRAY_A );

		// Generate the data of the db table.
		foreach ( $data_a as $record ) {

			$out .= "<$db_table_name>";

			// Get all the indexes of the $data array.
			$record_keys = array_keys( $record );

			// Cycle through all the indexes of the single record and create all the XML tags.
			foreach ( $record_keys as $key ) {
				$out .= '<' . $key . '>' . esc_attr( $record[ $key ] ) . '</' . $key . '>';
			}

			$out .= "</$db_table_name>";

		}

		return $out;
	}

	/**
	 * Objects as a value are set to empty strings. This prevent generating notices with the methods of the wpdb class.
	 *
	 * @param array $data An array which includes objects that should be converted to a empty strings.
	 * @return mixed An array where the objects have been replaced with empty strings.
	 */
	public function replace_objects_with_empty_strings( $data ) {

		foreach ( $data as $key => $value ) {
			if ( gettype( $value ) === 'object' ) {
				$data[ $key ] = '';
			}
		}

		return $data;
	}

	/*
	 * Verifies the number of records available in the database tables of the plugin. If there is at least one record
	 * returns true. Otherwise, returns false.
	 *
	 * @return bool
	 */

	/**
	 * Verifies the number of records available in the database tables of the plugin. If there is at least one record
	 *  returns true. Otherwise, returns false.
	 *
	 * @return bool
	 */
	public function plugin_has_data() {

		global $wpdb;

		$table_name    = $wpdb->prefix . $this->get( 'slug' ) . '_tickers';
		$tickers_items = $wpdb->get_var( "SELECT COUNT(*) FROM $table_name" );

		$table_name    = $wpdb->prefix . $this->get( 'slug' ) . '_featured_news';
		$featured_news = $wpdb->get_var( "SELECT COUNT(*) FROM $table_name" );

		$table_name   = $wpdb->prefix . $this->get( 'slug' ) . '_sliding_news';
		$sliding_news = $wpdb->get_var( "SELECT COUNT(*) FROM $table_name" );

		$total_items = intval( $tickers_items, 10 ) +
						intval( $featured_news, 10 ) +
						intval( $sliding_news, 10 );

		if ( $total_items > 0 ) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Delete all the records of the database tables used by the plugin.
	 *
	 * @return void
	 */
	public function reset_plugin_database_tables() {

		// Delete data in 'tickers' table.
		global $wpdb;
		$table_name = $wpdb->prefix . $this->get( 'slug' ) . '_tickers';
		$wpdb->query( "DELETE FROM $table_name" );

		// Delete data in 'featured_news' table.
		global $wpdb;
		$table_name = $wpdb->prefix . $this->get( 'slug' ) . '_featured_news';
		$wpdb->query( "DELETE FROM $table_name" );

		// Delete data in 'sliding_news' table.
		global $wpdb;
		$table_name = $wpdb->prefix . $this->get( 'slug' ) . '_sliding_news';
		$wpdb->query( "DELETE FROM $table_name" );
	}

	/**
	 * Reset the plugin options.
	 *
	 * Set the initial value to all the plugin options.
	 */
	public function reset_plugin_options() {

		$options = $this->get( 'options' );
		foreach ( $options as $option_name => $default_option_value ) {
			update_option( $option_name, $default_option_value );
		}
	}
}
